const { v4: uuidv4 } = require('uuid');

/**
 * @swagger
 * tags:
 *   name: Users
 *   description: API for user management
 */

/**
 * @swagger
 * /users:
 *   get:
 *     summary: Get all users
 *     description: A list of users
 *     tags:
 *       - Users
 *     responses:
 *       200:
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   name:
 *                     type: string
 */

const express = require('express');
const router = express.Router();
const User = require('../models/Category');


// Get all users from MongoDB
router.get('/', async (req, res) => {
    try {
        const users = await User.find().select('-password -__v'); // Exclude password field
        res.json(users);
    } catch (err) {
        res.status(500).json({ error: 'Failed to fetch users' });
    }
});

router.post('/', async (req, res) => {
    try {
        console.log('Creating user with data:', req.body);
        const { name, email, password } = req.body;
        if (!name || !email || !password) {
            return res.status(400).json({ error: 'Name, email, and password are required' });
        }

        const user = new User({ name, email, password });
        await user.save();
        res.status(201).json({ created: true, userId: user._id });
    } catch (err) {
        res.status(500).json({ error: err.message || 'Failed to create user' });
    }
})

/**
 * @swagger
 * /users/{id}:
 *   get:
 *     summary: Get a user by ID
 *     tags:
 *       - Users
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Numeric ID of the user to get
 *     responses:
 *       200:
 *         description: A user object
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 name:
 *                   type: string
 *       404:
 *         description: User not found
 */

// Get a user by MongoDB _id
router.get('/:id', async (req, res) => {
    try {
        const user = await User.findById(req.params.id);
        if (user) {
            res.json(user);
        } else {
            res.status(404).json({ error: 'User not found' });
        }
    } catch (err) {
        res.status(400).json({ error: 'Invalid user ID' });
    }
});

/**
 * @swagger
 * /users/random/{count}:
 *   get:
 *     summary: Get a random list of users
 *     tags:
 *       - Users
 *     parameters:
 *       - in: path
 *         name: count
 *         schema:
 *           type: integer
 *         required: true
 *         description: Number of random users to generate
 *     responses:
 *       200:
 *         description: A list of random users
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   name:
 *                     type: string
 *       400:
 *         description: Invalid count parameter
 */

// Create and return random users in MongoDB (for demo/testing)
router.get('/random/:count', async (req, res) => {
    const count = parseInt(req.params.count, 10);
    if (isNaN(count) || count <= 0 || count > 100) {
        return res.status(400).json({ error: 'Invalid count parameter' });
    }

    try {
        // Create random users
        const users = [];
        for (let i = 0; i < count; i++) {
            const user = new User({
                name: `User ${Math.floor(Math.random() * 10000)}`,
                email: `user${Date.now()}${i}@example.com`,
                password: uuidv4(),
                role: Math.random() < 0.5 ? 'user' : 'admin'
            });
            await user.save();
            users.push(user);
        }
        res.json(users);
    } catch (err) {
        res.status(500).json({ error: `Failed to create random users: ${JSON.stringify(err)}` });
    }
});

/**
 * @swagger
 * /users/{id}/edit:
 *   get:
 *     summary: update a user by ID
 *     tags:
 *       - Users
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: Numeric ID of the user to update
 *     responses:
 *       200:
 *         description: A user object
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 name:
 *                   type: string
 *       404:
 *         description: User not found
 */
// Update a user by MongoDB _id
router.post('/:id/edit', async (req, res) => {
    try {
        console.log('Updating user with ID:', req.params.id, 'with data:', req.body);
        const user = await User.findByIdAndUpdate(
            req.params.id,
            req.body,
            { new: true }
        );
        if (user) {
            res.json(user);
        } else {
            res.status(404).json({ error: 'User not found' });
        }
    } catch (err) {
        res.status(400).json({ error: 'Invalid user ID' });
    }
});
module.exports = router;
